function z = imp2D(p)
% imp2D.m
%
% Computes implicit solution to nonlinear hillslope diffusion equation
%
% syntax: z = imp2D(p)
%
% where p is a struct array with the following fields:
% 
% p.dx     delta x (m)
% p.dy     delta x (m)
% p.dt     delta t (yr)
% p.Lx     x length of grid (m)
% p.Ly     y length of grid (m)
% p.T      total run time (yr)
% p.U      uplift rate (m/yr)
% p.K      transport coefficient (m^2/yr)
% p.Sc     critical gradient (m/m)
% p.prps   rho_rock/rho_soil
% p.doplot 1 to plot solution at each timestep, 0 not
%
%
% Copyright 2008 J. T. Perron

%% input parameters

% create local variables 
dx=p.dx;
dy=p.dy;
dt=p.dt;
T=p.T;
Lx=p.Lx;
Ly=p.Ly;

doplot = p.doplot; 

N = round(T/dt); % number of iterations to do

x = (0:dx:Lx);
y = (0:dy:Ly);
p.Nx = length(x); Nx=p.Nx;
p.Ny = length(y); Ny=p.Ny;


%% specify boundary points 
% 0 = both influx and outflux are permitted (but no flux across boundaries)
% 1 = no influx, no outflux, and no uplift (elevations are fixed)
% 2 = periodic (in x if on x boundary OR CORNER, in y if on y boundary)
% (NOT YET SUPPORTED -- THIS VERSION ONLY USES FIXED BOUNDARIES.)

% create boundary array
p.bdy = zeros(Ny,Nx);

% OPTIONS:

% 1. fixed x and y boundaries
p.bdy([1 Ny],:)=1; % y boundaries
p.bdy(:,[1 Nx])=1; % x boundaries


% % 1.5 add a notch:
% p.bdy(round(Ny/2):Ny,round(Nx/2)) = 1;

% 2. radially symmetric cone
rad=min(Lx,Ly)/2;
[xm ym] = meshgrid(x,y);
xc=mean(x); yc=mean(y);
r=sqrt((xm-xc).^2 + (ym-yc).^2);
p.bdy(r>=rad)=1;

% END OPTIONS

p.bdy = p.bdy(:); % make column vector


%% initial conditions

% OPTIONS:

% % 1. flat
% z = zeros(Ny*Nx,1);

% 2. analytic solution
z = sol2D(r,rad,p.K,p.Sc,p.prps,p.U);
z = z(:);
z(z>0)=0;

% END OPTIONS

%% set up plot
if doplot
    ve = 1; % vertical exaggeration
    zmax = 100; % maximum z value
    figure;
    ax = axes(...
    'Projection','perspective',...
    'DataAspectRatio',1./[dx dy ve]...
    );
    DrawPlot(ax,reshape(z,Ny,Nx),0,0,zmax)
end

%% main iteration loop
t=0;

tic
for n=1:N
       
    t=t+dt;
        
    % update elevations
    z = IMP(z,p);
    
    % update plot
    if doplot
        DrawPlot(ax,reshape(z,Ny,Nx),n,t,zmax)
    end
    
end
toc

%% reshape return argument
z = reshape(z,Ny,Nx);
